<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class CarController extends Controller
{
    public function index(Request $request)
    {

        $status = false;
        $search = $request->search && trim($request->search) && $request->search != '0' ? str_replace(' ', '%', $request->search) : '';
        $queryData = json_decode($request->queryData, true);
        if ($request->status) {
            $status = json_decode($request->status);
        }

        $data = DB::table('cars')
            ->join('car_brands', 'cars.brand_id', '=', 'car_brands.id')
            ->join('car_models', 'cars.model_id', '=', 'car_models.id')
            ->leftJoin('vendors', 'vendors.id', '=', 'cars.vendor_id')
            ->when($status, function ($query) use ($status) {
                $query->whereIn('cars.status', (array) $status);
            })
            ->when($request->featured, function ($query) use ($request) {
                $query->where('cars.featured', true);
            })
            ->when(! empty($search), function ($query) use ($search) {
                $query->where('car_brands.name', 'LIKE', "%{$search}%")
                    ->orWhere('car_models.name', 'LIKE', "%{$search}%")
                    ->orWhere('cars.slug', 'LIKE', "%{$search}%");
            })
            // selectedYear
            ->when(! empty($queryData['selectedYear']), function ($query) use ($queryData) {
                $query->where('cars.year', $queryData['selectedYear']);
            })

            // selectedBrand
            ->when(! empty($queryData['selectedBrand']), function ($query) use ($queryData) {
                $query->where('car_brands.name', $queryData['selectedBrand']);
            })

            // RelatedCars
            ->when(! empty($queryData['RelatedCars']), function ($query) use ($queryData) {
                $query->where('car_brands.name', $queryData['RelatedCars']);
            })

            // Subject
            ->when(! empty($queryData['selectedModel']), function ($query) use ($queryData) {
                $query->where('car_models.name', $queryData['selectedModel']);
            })

            // selectedBody
            ->when(! empty($queryData['selectedBody']), function ($query) use ($queryData) {
                $query->whereIn('cars.body_type', (array) $queryData['selectedBody']);
            })

            // flashSales
            ->when(!empty($queryData['flashSales']), function ($query) use ($queryData) {
                $query->where('cars.flash_sale', true);
            })

            // priceRange
            ->when(
                !empty($queryData['priceRange']) && count($queryData['priceRange']) === 2,
                function ($query) use ($queryData) {
                    [$min, $max] = $queryData['priceRange'];

                    if (!is_null($min)) {
                        $query->where('cars.price', '>=', $min);
                    }

                    if (!is_null($max)) {
                        $query->where('cars.price', '<=', $max);
                    }
                }
            )
            ->select(
                'cars.*',
                'car_brands.name as brand',
                'car_models.name as model',
                'vendors.name as vendor',
                DB::raw('(
                    SELECT car_images.image_path
                    FROM car_images
                    WHERE car_images.car_id = cars.id
                    ORDER BY car_images.id ASC
                    LIMIT 1
                ) as firstImage')
            )
            ->when($status && $status != '0', function ($query) use ($status) {
                $query->orderBy('cars.id', 'DESC')
                    ->limit(20);
            })
            ->when($request->offset, function ($query) use ($request) {
                $query->offset(($request->offset - 1) * 16)
                    ->orderBy('cars.id', 'DESC')
                    ->limit(16);
            })->when(!empty($queryData['flashSales']), function ($query) {
                $query->inRandomOrder()
                    ->limit(8);
            })->when(! empty($queryData['RelatedCars']), function ($query) use ($queryData) {
                $query->orderByRaw('RAND()')
                    ->limit(8);
            })->when($request->featured, function ($query) use ($request) {
                $query->limit(12);
            })
            ->get();

        return $this->sendResponse(['data' => $data], 'Cars retrieved', 200);
    }


    public function getPromoCars(Request $request)
    {

        $status = false;
        if ($request->status) {
            $status = json_decode($request->status);
        }

        $data = DB::table('cars')
            ->join('car_brands', 'cars.brand_id', '=', 'car_brands.id')
            ->join('car_models', 'cars.model_id', '=', 'car_models.id')
            ->leftJoin('vendors', 'vendors.id', '=', 'cars.vendor_id')
            ->when($status, function ($query) use ($status) {
                $query->whereIn('cars.status', (array) $status);
            })
            ->select(
                'cars.*',
                'car_brands.name as brand',
                'car_models.name as model',
                'vendors.name as vendor',
                DB::raw('(
                    SELECT car_images.image_path
                    FROM car_images
                    WHERE car_images.car_id = cars.id
                    ORDER BY car_images.id ASC
                    LIMIT 1
                ) as firstImage')
            )->orderBy('cars.price', 'ASC')
            ->limit(10)
            ->get();

        return $this->sendResponse(['data' => $data], 'Cars retrieved', 200);
    }

    public function show($id)
    {
        $car = DB::table('cars')
            ->join('car_brands', 'cars.brand_id', '=', 'car_brands.id')
            ->join('car_models', 'cars.model_id', '=', 'car_models.id')
            ->leftJoin('vendors', 'vendors.id', '=', 'cars.vendor_id')
            ->where(function ($q) use ($id) {
                $q->where('cars.id', $id)
                    ->orWhere('cars.slug', $id);
            })
            ->select(
                'cars.*',
                'car_brands.name as brand',
                'car_models.name as model',
                'vendors.name as vendor',
                DB::raw('(
                    SELECT car_images.image_path
                    FROM car_images
                    WHERE car_images.car_id = cars.id
                    ORDER BY car_images.id ASC
                    LIMIT 1
                ) as firstImage')
            )->first();

        if (!$car) {
            return $this->sendError('Car not found', 'Not Found', 404);
        }

        $images = DB::table('car_images')
            ->where('car_id', $car->id)
            ->get();

        return $this->sendResponse([
            'car' => $car,
            'images' => $images
        ], 'Car retrieved', 200);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'slug' => 'required|string|unique:cars,slug',
            'brand_id' => 'required|exists:car_brands,id',
            'model_id' => 'required|exists:car_models,id',
            'year' => 'required|digits:4',
            'price' => 'required|numeric',
            'carImages' => 'required|array|min:1|max:10',
            'carImages.*' => 'image|mimes:jpg,jpeg,png,webp|max:5120',
            'location' => 'required|string',
        ]);

        if ($validator->fails()) {
            return $this->sendError($validator->errors(), 'Validation Error', 422);
        }

        DB::beginTransaction();

        try {
            $carId = DB::table('cars')->insertGetId([
                'slug' => Str::slug($request->slug),
                'vendor_id' => $request->vendor_id ?? null,
                'brand_id' => $request->brand_id,
                'model_id' => $request->model_id,
                'year' => $request->year,
                'price' => $request->price,
                'mileage' => $request->mileage ?? null,
                'transmission' => $request->transmission,
                'fuel_type' => $request->fuel_type ?? null,
                'engine_size' => $request->engine_size ?? null,
                'body_type' => $request->body_type ?? null,
                'color' => $request->color,
                'status' => $request->status ?? 'Available',
                'featured' => $request->featured ?? 0,
                'flash_sale' => $request->flash_sale ?? 0,
                'condition' => $request->condition,
                'registered' => $request->registered,
                'negotiable' => $request->negotiable ?? 1,
                'swap_allowed' => $request->swap_allowed,
                'description' => $request->description,
                'features' => json_encode($request->features),
                'location' => $request->location,
                'created_at' => now(),
            ]);

            if ($carId) {
                DB::table('cars')
                    ->where('id', $carId)->update([
                        'slug' => Str::slug($request->slug . ' ' . $carId)
                    ]);
            }

            if ($request->hasFile('carImages')) {
                foreach ($request->file('carImages') as $image) {

                    $path = $image->store('images/car_images', 'public');

                    DB::table('car_images')->insert([
                        'car_id' => $carId,
                        'image_path' => $path,
                        'created_at' => now(),
                    ]);
                }
            }

            DB::commit();

            return $this->sendResponse(['data' => $carId], 'Car created', 201);
        } catch (\Exception $e) {
            DB::rollBack();

            return $this->sendError($e->getMessage(), 'Server Error', 500);
        }
    }


    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required|integer',
            'slug' => 'required|string|unique:cars,slug,' . $request->id,
            'brand_id' => 'required|exists:car_brands,id',
            'model_id' => 'required|exists:car_models,id',
            'year' => 'required|digits:4',
            'price' => 'required|numeric',
            'carImages' => 'nullable|array|min:1|max:10',
            'carImages.*' => 'image|mimes:jpg,jpeg,png,webp|max:5120',
            'location' => 'required|string',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        DB::beginTransaction();

        try {
            $carId = DB::table('cars')
                ->where('id', $request->id)->update([
                    'slug' => Str::slug($request->slug . ' ' . $request->id),
                    'vendor_id' => $request->vendor_id ?? null,
                    'brand_id' => $request->brand_id,
                    'model_id' => $request->model_id,
                    'year' => $request->year,
                    'price' => $request->price,
                    'mileage' => $request->mileage,
                    'transmission' => $request->transmission,
                    'fuel_type' => $request->fuel_type,
                    'engine_size' => $request->engine_size,
                    'body_type' => $request->body_type,
                    'color' => $request->color,
                    'status' => $request->status ?? 'Available',
                    'featured' => $request->featured ?? 0,
                    'flash_sale' => $request->flash_sale ?? 0,
                    'condition' => $request->condition,
                    'registered' => $request->registered,
                    'negotiable' => $request->negotiable ?? 1,
                    'swap_allowed' => $request->swap_allowed,
                    'description' => $request->description,
                    'features' => json_encode($request->features),
                    'location' => $request->location,
                    'updated_at' => now(),
                ]);


            if ($request->deleted_images) {
                foreach ($request->deleted_images as $id) {
                    DB::table('car_images')->where('id', $id)->delete();
                }
            }

            if ($request->hasFile('carImages')) {
                foreach ($request->file('carImages') as $image) {
                    $path = $image->store('images/car_images', 'public');
                    DB::table('car_images')->insert([
                        'car_id' => $request->id,
                        'image_path' => $path,
                        'created_at' => now(),
                    ]);
                }
            }

            DB::commit();

            return $this->sendResponse(['data' => $carId], 'Car created', 201);
        } catch (\Exception $e) {
            DB::rollBack();

            return $this->sendError($e->getMessage(), 'Server Error', 500);
        }
    }

    public function toggleStatus(Request $request)
    {
        $car = DB::table('cars')->where('id', $request->id)->first();

        if (!$car) {
            return $this->sendError(
                'Vendor not found',
                'Not Found',
                404
            );
        }

        DB::table('cars')
            ->where('id', $car->id)
            ->update([
                'status' => $request->status
            ]);

        return $this->sendResponse(
            ['data' => $request->id],
            'Vendor status updated',
            200
        );
    }

    public function analytics()
    {
        /**
         * 1️⃣ Monthly cars & sales (last 6 months)
         */
        $monthlyRaw = DB::table('cars')
            ->select(
                DB::raw('MONTH(created_at) as month_num'),
                DB::raw('MONTHNAME(created_at) as month'),
                DB::raw('COUNT(id) as cars'),
                DB::raw("SUM(CASE WHEN status = 'Sold' THEN 1 ELSE 0 END) as sales")
            )
            ->where('created_at', '>=', now()->subMonths(6))
            ->groupBy('month_num', 'month')
            ->orderBy('month_num')
            ->get();

        $monthlyData = $monthlyRaw->map(function ($item) {
            return [
                'month' => substr($item->month, 0, 3), // Jan, Feb...
                'cars' => (int) $item->cars,
                'sales' => (int) $item->sales,
            ];
        });

        /**
         * 2️⃣ Recent Cars
         */
        $recentCars = DB::table('cars as c')
            ->join('car_brands as b', 'c.brand_id', '=', 'b.id')
            ->join('car_models as m', 'c.model_id', '=', 'm.id')
            ->select(
                'c.id',
                'b.name as brand',
                'm.name as model',
                'c.price',
                'c.status',
                'c.created_at'
            )
            ->orderBy('c.created_at', 'desc')
            ->limit(5)
            ->get()
            ->map(function ($car) {
                return [
                    'id' => (string) $car->id,
                    'brand' => $car->brand,
                    'model' => $car->model,
                    'price' => '₦' . number_format($car->price, 0),
                    'date' => Carbon::parse($car->created_at)->diffForHumans(),
                    'status' => ucfirst($car->status),
                ];
            });

        /**
         * 3️⃣ Recent Activity (derived events)
         */
        $recentActivity = collect();

        // New cars
        $newCars = DB::table('cars as c')
            ->join('car_brands as b', 'c.brand_id', '=', 'b.id')
            ->join('car_models as m', 'c.model_id', '=', 'm.id')
            ->select(
                'c.id',
                'b.name as brand',
                'm.name as model',
                'c.created_at'
            )
            ->orderBy('c.created_at', 'desc')
            ->limit(3)
            ->get();

        foreach ($newCars as $car) {
            $recentActivity->push([
                'id' => 'car-' . $car->id,
                'action' => 'New car added',
                'item' => "{$car->brand} {$car->model}",
                'time' => Carbon::parse($car->created_at)->diffForHumans(),
            ]);
        }

        // Sold cars
        $soldCars = DB::table('cars as c')
            ->join('car_brands as b', 'c.brand_id', '=', 'b.id')
            ->join('car_models as m', 'c.model_id', '=', 'm.id')
            ->where('c.status', 'Sold')
            ->orderBy('c.updated_at', 'desc')
            ->limit(2)
            ->get();

        foreach ($soldCars as $car) {
            $recentActivity->push([
                'id' => 'sold-' . $car->id,
                'action' => 'Car sold',
                'item' => "{$car->brand} {$car->model}",
                'time' => Carbon::parse($car->updated_at)->diffForHumans(),
            ]);
        }


        $currentMonthStart = Carbon::now()->startOfMonth();
        $lastMonthStart = Carbon::now()->subMonth()->startOfMonth();
        $lastMonthEnd = Carbon::now()->subMonth()->endOfMonth();

        /**
         * Helper to calculate percentage change
         */
        $percentageChange = function ($current, $previous) {
            if ($previous == 0) {
                return $current > 0 ? 100 : 0;
            }

            return round((($current - $previous) / $previous) * 100);
        };

        /**
         * 1️⃣ Total Cars
         */
        $totalCars = DB::table('cars')->count();

        $totalCarsLastMonth = DB::table('cars')
            ->whereBetween('created_at', [$lastMonthStart, $lastMonthEnd])
            ->count();

        $totalCarsChange = $percentageChange($totalCars, $totalCarsLastMonth);

        /**
         * 2️⃣ Available Cars
         */
        $availableCars = DB::table('cars')
            ->where('status', 'Available')
            ->count();

        $availableCarsLastMonth = DB::table('cars')
            ->where('status', 'Available')
            ->whereBetween('created_at', [$lastMonthStart, $lastMonthEnd])
            ->count();

        $availableCarsChange = $percentageChange($availableCars, $availableCarsLastMonth);

        /**
         * 3️⃣ Sold Cars
         */
        $soldCars = DB::table('cars')
            ->where('status', 'Sold')
            ->count();

        $soldCarsLastMonth = DB::table('cars')
            ->where('status', 'Sold')
            ->whereBetween('updated_at', [$lastMonthStart, $lastMonthEnd])
            ->count();

        $soldCarsChange = $percentageChange($soldCars, $soldCarsLastMonth);

        /**
         * 4️⃣ Active Vendors
         */
        $activeVendors = DB::table('vendors')
            ->where('is_active', true)
            ->count();

        $activeVendorsLastMonth = DB::table('vendors')
            ->where('is_active', true)
            ->whereBetween('created_at', [$lastMonthStart, $lastMonthEnd])
            ->count();

        $activeVendorsChange = $percentageChange($activeVendors, $activeVendorsLastMonth);

        /**
         * KPI Cards
         */
        $stats = [
            [
                'title' => 'Total Cars',
                'value' => $totalCars,
                'change' => $totalCarsChange,
            ],
            [
                'title' => 'Available Cars',
                'value' => $availableCars,
                'change' => $availableCarsChange,
            ],
            [
                'title' => 'Sold Cars',
                'value' => $soldCars,
                'change' => $soldCarsChange,
            ],
            [
                'title' => 'Active Vendors',
                'value' => $activeVendors,
                'change' => $activeVendorsChange,
            ],
        ];

        /**
         * Final Response
         */
        return $this->sendResponse([
            'stats' => $stats,
            'monthlyData' => $monthlyData,
            'recentCars' => $recentCars,
            'recentActivity' => $recentActivity->sortByDesc('time')->values(),
        ], 'Dashboard analytics retrieved', 200);
    }

    public function destroy($id)
    {
        $vendor = DB::table('cars')->where('id', $id)->first();

        if (!$vendor) {
            return $this->sendError(
                'Vendor not found',
                'Not Found',
                404
            );
        }

        DB::table('cars')->where('id', $id)->delete();

        return $this->sendResponse(
            ['data' => $id],
            'cars deleted successfully',
            200
        );
    }
}
