<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class TestimonialController extends Controller
{
    /**
     * Get all testimonials
     */
    public function index()
    {
        $data = DB::table('testimonials')
            ->orderBy('created_at', 'desc')
            ->get();

        return $this->sendResponse(
            ['data' => $data],
            'testimonials retrieved',
            200
        );
    }

    /**
     * Get single testimonials
     */
    public function show($id)
    {
        $vendor = DB::table('testimonials')
            ->where('id', $id)
            ->first();

        if (!$vendor) {
            return $this->sendError(
                'Vendor not found',
                'Not Found',
                404
            );
        }

        return $this->sendResponse(
            ['data' => $vendor],
            'Vendor retrieved',
            200
        );
    }

    /**
     * Create testimonials
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'text' => 'required|string|max:500',
            'rating' => 'required|integer|max:5',
            'car' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        try {
            $id = DB::table('testimonials')->insertGetId([
                'name' => $request->name,
                'text' => $request->text,
                'rating' => $request->rating ?? 5,
                'location' => $request->location,
                'car' => $request->car,
                'created_at' => now(),
            ]);

            return $this->sendResponse(
                ['data' => $id],
                'Vendor created successfully',
                201
            );
        } catch (\Illuminate\Validation\ValidationException $e) {


            return $this->sendError($e->errors()[array_key_first($e->errors())][0], 'Server Error', 500);
        }
    }

    /**
     * Update testimonials
     */
    public function update(Request $request)
    {
        $vendor = DB::table('testimonials')->where('id', $request->id)->first();

        if (!$vendor) {
            return $this->sendError(
                'Vendor not found',
                'Not Found',
                404
            );
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'text' => 'required|string|max:500',
            'rating' => 'required|integer|max:5',
            'car' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        try {
            DB::table('testimonials')
                ->where('id', $request->id)
                ->update([
                    'name' => $request->name,
                    'text' => $request->text,
                    'rating' => $request->rating ?? 5,
                    'location' => $request->location,
                    'car' => $request->car,
                ]);

            return $this->sendResponse(
                ['data' => $request->id],
                'Vendor updated successfully',
                200
            );
        } catch (\Illuminate\Validation\ValidationException $e) {


            return $this->sendError($e->errors()[array_key_first($e->errors())][0], 'Server Error', 500);
        }
    }


    /**
     * Delete testimonials
     */
    public function destroy($id)
    {
        $vendor = DB::table('testimonials')->where('id', $id)->first();

        if (!$vendor) {
            return $this->sendError(
                'Vendor not found',
                'Not Found',
                404
            );
        }

        DB::table('testimonials')->where('id', $id)->delete();

        return $this->sendResponse(
            ['data' => $id],
            'Vendor deleted successfully',
            200
        );
    }
}
