<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class UserController extends Controller
{
    public function store(Request $request)
    {
        $input = $request->all();

        $validator = Validator::make($input, [
            'organizations' => 'required|array',
            'sname' => 'required|string|max:255', // Ensure surname is required and a string
            'oname' => 'required|string|max:255', // Ensure other name is required and a string
            'username' => 'required|alpha_dash|max:255|unique:users',
            'email' => $request->has('email') && $request->email != ''
                ? 'required|email|unique:users,email'
                : 'nullable|email', // Ensure nullable email is valid
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        DB::beginTransaction();

        $user_id = ! $this->checkJWTAuth() ? 0 : $this->JWTAuth()->id;
        $input['password'] = bcrypt('12345678');
        $user = DB::table('users')->insertGetId([
            'uuid' => Str::uuid(),
            'sname' => $input['sname'],
            'oname' => $input['oname'],
            'username' => $input['username'],
            'email' => $input['email'] ?? null,
            'password' => $input['password'],
            'role' => 'learner',
            'created_at' => \Carbon\Carbon::now(),
        ]);

        if (is_array($input['organizations'])) {
            $brorganizations = $input['organizations'];
        } else {
            $brorganizations = json_decode($input['organizations'], true);
        }

        foreach ($brorganizations as $org_id) {
            DB::table('organization_user')->insert([
                'organization_id' => $org_id,
                'user_id' => $user,
                'role' => 'student',
                'created_by' => $user_id,
                'created_at' => now(),
            ]);
        }

        if (! $user || ! $brorganizations) {
            DB::rollBack();
            $error = 'something went wrong';

            return $this->sendError($error, 'Validation Error', 422);
        } else {
            DB::commit();
            $success['data'] = $user;

            return $this->sendResponse($success, 'successfully', 201);
        }
    }

    public function importStudents(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|file|mimes:csv,txt|max:2048',
            'organizations' => 'required|array',
        ]);

        if ($validator->fails()) {
            return $this->sendError($validator->errors()->first(), 'Validation Error', 422);
        }

        $file = $request->file('file');
        $handle = fopen($file->getRealPath(), 'r');

        if (! $handle) {
            return $this->sendError('Cannot read file', 'Cannot read file', 400);
        }

        DB::beginTransaction();
        $creator_id = $this->checkJWTAuth() ? $this->JWTAuth()->id : 0;

        try {
            $insertData = [];
            $rowNumber = 0;

            while (($row = fgetcsv($handle)) !== false) {
                $rowNumber++;
                if (count($row) < 4) {
                    throw new \Exception("Row {$rowNumber}: Invalid column count");
                }

                $data = [
                    'username' => trim($row[0]),
                    'sname' => trim($row[1]),
                    'oname' => trim($row[2]),
                    'email' => trim($row[3]),
                ];

                $rowValidator = Validator::make($data, [
                    'username' => 'required|string|max:255|unique:users,username',
                    'sname' => 'required|string|max:255',
                    'oname' => 'required|string|max:255',
                    'email' => 'nullable|email|unique:users,email',
                ]);

                if ($rowValidator->fails()) {
                    throw new \Exception("Row {$rowNumber}: " . $rowValidator->errors()->first());
                }

                $insertData[] = [
                    'uuid' => Str::uuid(),
                    'username' => $data['username'],
                    'sname' => $data['sname'],
                    'oname' => $data['oname'],
                    'email' => $data['email'] ?? null,
                    'password' => bcrypt('12345678'),
                    'created_at' => now(),
                ];
            }

            fclose($handle);

            // Bulk insert
            DB::table('users')->insert($insertData);

            // Get the inserted user IDs
            $userIds = DB::table('users')
                ->whereIn('username', array_column($insertData, 'username'))
                ->pluck('id')
                ->toArray();

            // Attach users to organizations
            $organizations = $request->input('organizations'); // should already be array if you use `organizations[]`
            $orgUserData = [];

            foreach ($userIds as $userId) {
                foreach ($organizations as $orgId) {
                    $orgUserData[] = [
                        'organization_id' => $orgId,
                        'user_id' => $userId,
                        'role' => 'student',
                        'created_by' => $creator_id,
                        'created_at' => now(),
                    ];
                }
            }

            DB::table('organization_user')->insert($orgUserData);
            DB::commit();

            return $this->sendResponse(['data' => count($insertData)], 'Users imported successfully', 201);
        } catch (\Exception $e) {
            DB::rollBack();

            return $this->sendError($e->getMessage(), 'Import failed', 422);
        }
    }

    public function getUserDetails(Request $request)
    {
        $user_id = ! $this->checkJWTAuth() ? 0 : $this->JWTAuth()->id;
        $details = DB::table('users')
            ->distinct()
            ->leftJoin('user_details', 'user_details.user_id', '=', 'users.id')
            ->leftJoin('user_assets', 'user_assets.user_id', '=', 'users.id')
            ->leftJoin('seller_branch_assignments', function ($join) use (
                $user_id
            ) {
                $join

                    ->where(
                        'seller_branch_assignments.seller_id',
                        '=',
                        $user_id
                    )
                    ->whereNull('seller_branch_assignments.deleted_at')
                    ->limit(1);
            })
            ->leftJoin(
                'branches',
                'branches.id',
                '=',
                'seller_branch_assignments.branch_id'
            )
            ->where('users.id', $request->user_id)
            ->orWhere('username', $request->user_id)
            ->select(
                'users.*',
                'user_details.about_user',
                'user_assets.profile_img',
                'user_assets.cover_img',
                'user_assets.thumbnail_cover_img',
                'seller_branch_assignments.branch_id',
                'branches.branch_name'
            )
            ->first();
        if (! $details) {
            $error = 'something went wrong';

            return $this->sendError($error, 'Validation Error', 422);
        } else {
            $success['data'] = $details;

            return $this->sendResponse($success, 'successfully', 201);
        }
    }

    public function getAvailUsers(Request $request)
    {
        $image = DB::table('users')
            ->whereNull('deleted_at')
            ->select('*')
            ->orderBy('id', 'desc')
            ->get();

        if (! $image) {
            $error = 'something went wrong';

            return $this->sendError($error, 'Validation Error', 422);
        } else {
            $success['data'] = $image;

            return $this->sendResponse($success, 'successfully', 201);
        }
    }

    public function editUser(Request $request)
    {
        $input = $request->all();

        $validator = Validator::make($input, [
            'sname' => 'required|string|max:255', // Ensure surname is required and a string
            'oname' => 'required|string|max:255', // Ensure other name is required and a string
            'username' => $request->old_username != $request->username
                ? 'required|string|max:255|unique:users'
                : 'nullable', // Remove invalid `username` rule
            'email' => $request->has('email') && $request->email != ''
                ? 'required|email|unique:users'
                : 'nullable|email', // Ensure nullable email is valid
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }
        $user_id = ! $this->checkJWTAuth() ? 0 : $this->JWTAuth()->id;
        DB::beginTransaction();

        $branch_id = DB::table('users')
            ->where('id', $request->id)
            ->update([
                'sname' => $input['sname'],
                'oname' => $input['oname'],
                'username' => $input['username'],
                'user_type' => $input['user_type'] ?? DB::raw('user_type'),
                'updated_at' => \Carbon\Carbon::now(),
            ]);

        if (! $branch_id) {
            DB::rollBack();
            $error = 'something went wrong';

            return $this->sendError($error, 'Validation Error', 422);
        } else {
            DB::commit();
            $success['data'] = $branch_id;

            return $this->sendResponse($success, 'successfully', 201);
        }
    }

    public function deleteUsers(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'selectedUsers' => 'required',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }
        $selectedUsers = json_decode($request->selectedUsers);
        if (is_array($selectedUsers)) {
            foreach ($selectedUsers as $val) {
                $ques_id = DB::table('users')
                    ->where('id', $val->id)
                    ->update([
                        'deleted_at' => \Carbon\Carbon::now(),
                    ]);
            }
        }
        if ($ques_id) {
            DB::commit();
            $success['data'] = $ques_id;

            return $this->sendResponse($success, 'successful', 201);
        } else {
            DB::rollBack();
            $error = 'something went wrong';

            return $this->sendError($error, 'Validation Error', 422);
        }
    }

    public function getSitePages(Request $request)
    {
        $settings_data = [];
        $settings = DB::table('settings')
        ->whereIn('key', (array) ['about','contact','privacy','faq'])
            ->get();



        if ($settings) {
            foreach ($settings as $setting) {
                $settings_data[$setting->key] = $setting->value;
            }
        }

        if (!$settings_data) {
            $error['data'] = 'something went wrong';

            return $this->sendError($error, 'Validation Error', 422);
        } else {
            $success['data'] = $settings_data;

            return $this->sendResponse($success, 'successfully', 201);
        }
    }


    public function getSiteInfo(Request $request)
    {
        $settings_data = [];
        $settings = DB::table('settings')
        ->whereNotIn('key', (array) ['about','contact','privacy','faq'])
            ->get();

        $priceGroups = DB::table('cars')
            ->select(
                DB::raw("
            CASE
                WHEN price BETWEEN 0 AND 2000000 THEN '₦0 - ₦2M'
                WHEN price BETWEEN 2000001 AND 4000000 THEN '₦2M - ₦4M'
                WHEN price BETWEEN 4000001 AND 6000000 THEN '₦4M - ₦6M'
                WHEN price BETWEEN 6000001 AND 10000000 THEN '₦6M - ₦10M'
                WHEN price BETWEEN 10000001 AND 13000000 THEN '₦10M - ₦13M'
                WHEN price BETWEEN 13000001 AND 15000000 THEN '₦13M - ₦15M'
                ELSE '₦15M+'
            END AS label
        "),
                DB::raw('
            CASE
                WHEN price BETWEEN 0 AND 2000000 THEN 0
                WHEN price BETWEEN 2000001 AND 4000000 THEN 2000001
                WHEN price BETWEEN 4000001 AND 6000000 THEN 4000001
                WHEN price BETWEEN 6000001 AND 10000000 THEN 6000001
                WHEN price BETWEEN 10000001 AND 13000000 THEN 10000001
                WHEN price BETWEEN 13000001 AND 15000000 THEN 13000001
                ELSE 15000001
            END AS min
        '),
                DB::raw('
            CASE
                WHEN price BETWEEN 0 AND 2000000 THEN 2000000
                WHEN price BETWEEN 2000001 AND 4000000 THEN 4000000
                WHEN price BETWEEN 4000001 AND 6000000 THEN 6000000
                WHEN price BETWEEN 6000001 AND 10000000 THEN 10000000
                WHEN price BETWEEN 10000001 AND 13000000 THEN 13000000
                WHEN price BETWEEN 13000001 AND 15000000 THEN 15000000
                ELSE NULL
            END AS max
        '),
                DB::raw('COUNT(*) as total')
            )
            ->groupBy('label', 'min', 'max')
            ->orderBy('min', 'ASC')
            ->get();



        if ($settings) {
            foreach ($settings as $setting) {
                if ($setting->key == 'hero_sliders') {
                    $value = json_decode($setting->value, true);

                    $settings_data[$setting->key] = is_array($value) ? $value : [];
                } else {
                    $settings_data[$setting->key] = $setting->value;
                }
            }
        }

        if (!$settings_data) {
            $error['data'] = 'something went wrong';

            return $this->sendError($error, 'Validation Error', 422);
        } else {
            $success['data'] = ['info' => $settings_data, 'priceGroups' => $priceGroups ?? []];

            return $this->sendResponse($success, 'successfully', 201);
        }
    }

    public function saveSiteInfo(Request $request)
    {
        $input = $request->all();

        $validator = Validator::make($input, [
            'site_name' => 'required|string|max:255',
            'site_phone' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        DB::beginTransaction();

        try {
            foreach ($input as $key => $value) {
                DB::table('settings')->updateOrInsert(
                    ['key' => $key],           // WHERE key = ?
                    ['value' => $value]        // UPDATE value
                );
            }

            DB::commit();

            return $this->sendResponse(
                ['data' => true],
                'Site settings updated successfully',
                200
            );
        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();

            return $this->sendError($e->errors()[array_key_first($e->errors())][0], 'Server Error', 500);
        }
    }


    public function saveSitePage(Request $request)
    {
        $input = $request->all();

        $validator = Validator::make($input, [
            'key' => 'required|string',
            'value' => 'required|string',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        try {

            DB::table('settings')->where('key', $request->key)->update(
                ['value' => $request->value]     // UPDATE value
            );

            return $this->sendResponse(
                ['data' => true],
                'Site settings updated successfully',
                200
            );
        } catch (\Illuminate\Validation\ValidationException $e) {


            return $this->sendError($e->errors()[array_key_first($e->errors())][0], 'Server Error', 500);
        }
    }


    public function saveSiteSlideImages(Request $request)
    {
        $input = $request->all();

        $validator = Validator::make($input, [
            'slideImages' => 'nullable|array|min:1|max:10',
            'slideImages.*' => 'image|mimes:jpg,jpeg,png,webp|max:5120',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        try {

            $images = [];

            $existingImages = [];

            if ($request->filled('images')) {
                $existingImages = json_decode($request->images, true);
            }

            $newImages = [];

            if ($request->hasFile('slideImages')) {
                foreach ($request->file('slideImages') as $image) {
                    $path = $image->store('images/hero_slide_images', 'public');
                    $newImages[] = $path;
                }
            }

            $finalImages = array_merge(
                array_map(fn ($p) => ['image_path' => $p], $existingImages),
                array_map(fn ($p) => ['image_path' => $p], $newImages)
            );

            DB::table('settings')
                ->where('key', 'hero_sliders')
                ->update([
                    'value' => json_encode($finalImages)
                ]);

            return $this->sendResponse(
                ['data' => true],
                'Site settings updated successfully',
                200
            );
        } catch (\Illuminate\Validation\ValidationException $e) {


            return $this->sendError($e->errors()[array_key_first($e->errors())][0], 'Server Error', 500);
        }
    }


    public function getHomePageData(Request $request)
    {
        $today = \Carbon\Carbon::today();
        $user_id = $this->JWTAuth()->id;
        $staffRoles = ['owner', 'admin', 'teacher'];

        $totalStudents = DB::table('organization_user')
            ->whereIn('organization_id', function ($sub) use ($user_id, $staffRoles) {
                $sub->select('organization_id')
                    ->from('organization_user')
                    ->where('user_id', $user_id)
                    ->where('role', 'student')
                    ->whereIn('role', $staffRoles);
            })
            ->distinct('user_id')
            ->count('user_id');

        $row = DB::table('courses')
            // 🔐 ACCESS RULES
            ->where(function ($q) use ($user_id, $staffRoles) {
                $q->where('courses.author_id', $user_id)
                    ->orWhereExists(function ($sub) use ($user_id, $staffRoles) {
                        $sub->from('organization_user as ou')
                            ->where('ou.user_id', $user_id)
                            ->whereIn('ou.role', $staffRoles)
                            ->whereRaw(
                                "JSON_CONTAINS(courses.organizations, JSON_OBJECT('id', CAST(ou.organization_id AS CHAR)))"
                            );
                    });
            })
            ->select(
                'courses.*'
            )
            ->get();

        $courseIds = $row->pluck('id');
        $topics = DB::table('topics')
            ->whereIn('course_id', $courseIds)
            ->orderBy('id')
            ->get()
            ->groupBy('course_id');

        $orgIds = collect();

        foreach ($row as $course) {
            $orgs = json_decode($course->organizations, true);

            if (is_array($orgs)) {
                foreach ($orgs as $org) {
                    if (isset($org['id'])) {
                        $orgIds->push((int) $org['id']);
                    }
                }
            }
        }

        $orgIds = $orgIds->unique()->values();

        /**
         * STEP 3: Fetch organizations
         */
        $organizations = DB::table('organizations')
            ->whereIn('id', $orgIds)
            ->select('id', 'name')
            ->get()
            ->keyBy('id');

        /**
         * STEP 4: Attach organizations to each level
         */
        $courses = $row->map(function ($course) use ($organizations, $topics) {

            // Attach organizations
            $orgs = json_decode($course->organizations, true);

            $course->organizations = collect($orgs)
                ->map(fn ($org) => $organizations[$org['id']] ?? null)
                ->filter()
                ->values();

            // Attach topics
            $course->topics = $topics[$course->id] ?? collect();

            return $course;
        });

        $totalLevels = DB::table('levels')
            ->whereNull('levels.deleted_at')
            // 🔐 ACCESS RULES
            ->where(function ($q) use ($user_id, $staffRoles) {
                $q->where('levels.author_id', $user_id)
                    ->orWhereExists(function ($sub) use ($user_id, $staffRoles) {
                        $sub->from('organization_user as ou')
                            ->where('ou.user_id', $user_id)
                            ->whereIn('ou.role', $staffRoles)
                            ->whereRaw(
                                "JSON_CONTAINS(levels.organizations, JSON_OBJECT('id', CAST(ou.organization_id AS CHAR)))"
                            );
                    });
            })->count('levels.id');

        $totalOrganizations = DB::table('organizations')
            ->where(function ($q) use ($user_id) {
                $q->where('organizations.author_id', $user_id)
                    ->orWhereExists(function ($sub) use ($user_id) {
                        $sub->select(DB::raw(1))
                            ->from('organization_user as ou')
                            ->whereColumn('ou.organization_id', 'organizations.id')
                            ->where('ou.user_id', $user_id)
                            ->whereIn('ou.role', ['owner', 'admin', 'teacher', 'student']);
                    });
            })
            ->count('organizations.id');

        $isStudent = DB::table('organization_user')
            ->where('user_id', $user_id)
            ->where('role', 'student')
            ->exists();

        $totalExams = DB::table('exams')
            ->where('exams.exam_type', 'exam')

            // ✅ Apply end_date filter ONLY for students
            ->when($isStudent, function ($q) {
                $q->where('exams.end_date', '>=', now());
            })

            // 🔐 Authorization
            ->where(function ($q) use ($user_id) {
                $q->where('exams.creator_id', $user_id)
                    ->orWhereExists(function ($sub) use ($user_id) {
                        $sub->select(DB::raw(1))
                            ->from('organization_exams as oe')
                            ->join('organization_user as ou', 'ou.organization_id', '=', 'oe.organization_id')
                            ->whereColumn('oe.exam_id', 'exams.id')
                            ->where('ou.user_id', $user_id)
                            ->whereIn('ou.role', ['owner', 'admin', 'teacher', 'student']);
                    });
            })
            ->count('exams.id');

        $lastAttempts = DB::table('attempts')
            ->select(DB::raw('MAX(id) as last_id'))
            ->groupBy('user_id');
        // Step 2: join back to get full attempt info
        $examPerformance = DB::table('attempts')
            ->joinSub($lastAttempts, 'la', function ($join) {
                $join->on('attempts.id', '=', 'la.last_id');
            })
            ->leftJoin('exams', 'exams.id', '=', 'attempts.exam_id')
            ->leftJoin('users', 'users.id', '=', 'attempts.user_id')
            ->whereNotNull('attempts.submitted_at')
            // 🔐 Authorization
            ->where(function ($q) use ($user_id, $staffRoles) {
                $q->where('exams.creator_id', $user_id)
                    ->orWhereExists(function ($sub) use ($user_id, $staffRoles) {
                        $sub->select(DB::raw(1))
                            ->from('organization_exams as oe')
                            ->join('organization_user as ou', 'ou.organization_id', '=', 'oe.organization_id')
                            ->whereColumn('oe.exam_id', 'exams.id')
                            ->where('ou.user_id', $user_id)
                            ->whereIn('ou.role', $staffRoles);
                    });
            })
            ->select(
                'attempts.id',
                'attempts.user_id',
                'attempts.score',
                'attempts.grading_summary',
                'exams.passing_score',
                'exams.title as exam_name',
                'users.username'
            )->offset(($request->offset - 1) * 20)
            ->limit(20)
            ->get();

        $homeData = [
            'totalStudents' => $totalStudents,
            'totalTopics' => $courses->sum(fn ($course) => $course->topics->count()),
            'totalCourses' => $courses->count(),
            'totalLevels' => $totalLevels,
            'totalOrganizations' => $totalOrganizations,
            'totalExams' => $totalExams,
            'examPerformance' => $examPerformance,
        ];

        if (! $homeData) {
            $error = 'something went wrong';

            return $this->sendError($error, 'Validation Error', 422);
        } else {
            $success['data'] = $homeData;

            return $this->sendResponse($success, 'successfully', 201);
        }
    }
}
