<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class VendorController extends Controller
{
    /**
     * Get all vendors
     */
    public function index()
    {
        $data = DB::table('vendors')
        ->leftJoin('cars as c', 'vendors.id', '=', 'c.vendor_id')
         ->select(
             'vendors.id',
             'vendors.name',
             'vendors.phone',
             'vendors.email',
             'vendors.is_active AS isActive',
             DB::raw('COUNT(c.id) as carsCount')
         ) ->groupBy(
             'vendors.id',
             'vendors.name',
             'vendors.phone',
             'vendors.email',
             'vendors.is_active',
         )
            ->orderBy('vendors.created_at', 'desc')
            ->get();

        return $this->sendResponse(
            ['data' => $data],
            'Vendors retrieved',
            200
        );
    }

    /**
     * Get single vendor
     */
    public function show($id)
    {
        $vendor = DB::table('vendors')
            ->where('id', $id)
            ->first();

        if (!$vendor) {
            return $this->sendError(
                'Vendor not found',
                'Not Found',
                404
            );
        }

        return $this->sendResponse(
            ['data' => $vendor],
            'Vendor retrieved',
            200
        );
    }

    /**
     * Create vendor
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:30',
            'email' => 'required|email|unique:vendors,email',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        try {
            $id = DB::table('vendors')->insertGetId([
                'name' => $request->name,
                'phone' => $request->phone,
                'email' => $request->email,
                'is_active' => $request->is_active ?? true,
                'created_at' => now(),
            ]);

            return $this->sendResponse(
                ['data' => $id],
                'Vendor created successfully',
                201
            );
        } catch (\Illuminate\Validation\ValidationException $e) {


            return $this->sendError($e->errors()[array_key_first($e->errors())][0], 'Server Error', 500);
        }
    }

    /**
     * Update vendor
     */
    public function update(Request $request)
    {
        $vendor = DB::table('vendors')->where('id', $request->id)->first();

        if (!$vendor) {
            return $this->sendError(
                'Vendor not found',
                'Not Found',
                404
            );
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:30',
            'email' => 'required|email|unique:vendors,email,' . $request->id,
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return $this->sendError(
                $validator->errors()->first(),
                'Validation Error',
                422
            );
        }

        try {
            DB::table('vendors')
                ->where('id', $request->id)
                ->update([
                    'name' => $request->name,
                    'phone' => $request->phone,
                    'email' => $request->email,
                    'is_active' => $request->is_active ?? $vendor->is_active,
                ]);

            return $this->sendResponse(
                ['data' => $request->id],
                'Vendor updated successfully',
                200
            );
        } catch (\Illuminate\Validation\ValidationException $e) {


            return $this->sendError($e->errors()[array_key_first($e->errors())][0], 'Server Error', 500);
        }
    }

    /**
     * Enable / Disable vendor
     */
    public function toggleStatus($id)
    {
        $vendor = DB::table('vendors')->where('id', $id)->first();

        if (!$vendor) {
            return $this->sendError(
                'Vendor not found',
                'Not Found',
                404
            );
        }

        DB::table('vendors')
            ->where('id', $id)
            ->update([
                'is_active' => !$vendor->is_active
            ]);

        return $this->sendResponse(
            ['data' => $id, 'is_active' => !$vendor->is_active],
            'Vendor status updated',
            200
        );
    }

    /**
     * Delete vendor
     */
    public function destroy($id)
    {
        $vendor = DB::table('vendors')->where('id', $id)->first();

        if (!$vendor) {
            return $this->sendError(
                'Vendor not found',
                'Not Found',
                404
            );
        }

        DB::table('vendors')->where('id', $id)->delete();

        return $this->sendResponse(
            ['data' => $id],
            'Vendor deleted successfully',
            200
        );
    }
}
